﻿using System;
using System.Runtime.CompilerServices;
using ConnectorLib.JSON;
using UnityEngine;

namespace CrowdControl
{
    public class GameStateManager
    {
        //Everything in the game-specific region will need to be changed for each game
    
        #region Game-Specific Code
        
        /// <summary>Whether to allow effects to be applied while the player is in a safe zone.</summary>
        public bool RespectSafeZones { get; set; }
        
        /// <summary>Checks if the player is in a safe zone, if safe zones are respected.</summary>
        /// <returns>True if the player is in a safe zone and safe zones are respected, false otherwise.</returns>
        public bool CheckSafeZoneIfNeeded() => RespectSafeZones && (Manager.main.player?.isInSafeZone ?? false);

        /// <summary>Checks if the game is in a state where effects can be applied.</summary>
        /// <param name="code">The effect codename the caller is intending to apply.</param>
        /// <returns>True if the game is in a state where the effect can be applied, false otherwise.</returns>
        /// <remarks>
        /// The <paramref name="code"/> parameter is not normally checked.
        /// Use this is you want to exempt certain effects from checks (e.g. debug or "fix-it" effects).
        /// </remarks>
        public bool IsReady(string code = "") => GetGameState() == GameState.Ready;

        /// <summary>Gets the current game state as it pertains to the firing of effects.</summary>
        /// <returns>The current game state.</returns>
        public GameState GetGameState()
        {
            try
            {
                SceneHandler scene = Manager.main.currentSceneHandler;
                if (scene == null) return GameState.Loading;
                if (scene.isTitle) return GameState.WrongMode;
                if (!scene.isInGame) return GameState.WrongMode;
                if (scene.isIntro) return GameState.WrongMode;
                if (scene.isOutro) return GameState.WrongMode;

                PlayerController player = Manager.main.player;
                if (player == null) return GameState.WrongMode;
                if (!player.isActiveAndEnabled) return GameState.WrongMode;
                if (player.isDyingOrDead) return GameState.BadPlayerState;
                if (RespectSafeZones && player.isInSafeZone) return GameState.SafeArea;

                if (scene.cutsceneIsPlaying) return GameState.Cutscene;
                if (!player.inputModule.InputEnabled) return GameState.Cutscene; //paused?
                
                //manager.menu next
                
                return GameState.Ready;
            }
            catch (Exception e)
            {
                Debug.LogError($"ERROR {e}");
                return GameState.Error;
            }
        }

        #endregion

        //Everything from here down is the same for every game - you probably don't need to change it

        #region General Code

        /// <summary>Reports the updated game state to the Crowd Control client.</summary>
        /// <param name="force">True to force the report to be sent, even if the state is the same as the previous state, false to only report the state if it has changed.</param>
        /// <returns>True if the data was sent successfully, false otherwise.</returns>
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public bool UpdateGameState(bool force = false) => UpdateGameState(GetGameState(), force);

        /// <summary>Reports the updated game state to the Crowd Control client.</summary>
        /// <param name="newState">The new game state to report.</param>
        /// <param name="force">True to force the report to be sent, even if the state is the same as the previous state, false to only report the state if it has changed.</param>
        /// <returns>True if the data was sent successfully, false otherwise.</returns>
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        public bool UpdateGameState(GameState newState, bool force) => UpdateGameState(newState, null, force);

        private GameState? _last_game_state;
        private readonly CrowdControl m_mod;
        public GameStateManager(CrowdControl mod)
        {
            m_mod = mod;
        }

        /// <summary>Reports the updated game state to the Crowd Control client.</summary>
        /// <param name="newState">The new game state to report.</param>
        /// <param name="message">The message to attach to the state report.</param>
        /// <param name="force">True to force the report to be sent, even if the state is the same as the previous state, false to only report the state if it has changed.</param>
        /// <returns>True if the data was sent successfully, false otherwise.</returns>
        public bool UpdateGameState(GameState newState, string? message = null, bool force = false)
        {
            if (force || (_last_game_state != newState))
            {
                _last_game_state = newState;
                return m_mod.Client.Send(new GameUpdate(newState, message));
            }

            return true;
        }

        #endregion
    }
}